<?php

declare(strict_types=1);

namespace MZoldak\EbayOrdersImport\Controller\EbayOrdersImport;

use Contao\MemberModel;
use Contao\System;
use Isotope\Model\Address;
use Isotope\Model\Product;
use Isotope\Model\ProductCollection\Order;
use MZoldak\EbayOrdersImport\Class\CSVDataProcessor;
use MZoldak\EbayOrdersImport\Class\EbayFieldsKeysFilter;
use MZoldak\EbayOrdersImport\Class\FieldsMappingExportImport;
use MZoldak\EbayOrdersImport\Interface\EbayOrderImportStep;
use MZoldak\EbayOrdersImport\Service\CSVFieldName;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\HttpFoundation\Request;

class ImportOptions implements EbayOrderImportStep {
    public function __construct(private Request $request) {
    }

    public function run(): array {
        $files = $this->request->files->all();

        $csvImportedData = $this->loadImportData($files['import_data']);
        [$newKeys, $removedKeys] = $this->compareKeysWithPreviousImport($csvImportedData->getHeaders());
        CSVFieldName::initialize();

        $savedFieldsMapping = (new FieldsMappingExportImport())->load();

        $this->saveNewKeys($csvImportedData->getHeaders());

        $fields = [
            'order' => EbayFieldsKeysFilter::getKeys('tl_iso_product_collection'),
            'member' => EbayFieldsKeysFilter::getKeys('tl_member'),
            'shipping_address' => EbayFieldsKeysFilter::getKeys('tl_iso_address'),
            'billing_address' => EbayFieldsKeysFilter::getKeys('tl_iso_address'),
            'product' => EbayFieldsKeysFilter::getKeys('tl_iso_product'),
        ];

        $tablesNames = [
            'order' => Order::getTable(),
            'member' => MemberModel::getTable(),
            'address' => Address::getTable(),
            'product' => Product::getTable(),
        ];

        return [
            'fields_mapping' => (array) $savedFieldsMapping,
            'headers' => $csvImportedData->getHeaders(),
            'data' => $csvImportedData->getData(),
            'table_fields' => $fields,
            'table_name' => $tablesNames,
            'new_keys' => $newKeys,
            'removed_keys' => $removedKeys,
        ];
    }

    private function loadImportData(UploadedFile $file) {
        $importData = new CSVDataProcessor($file->getRealPath());

        if (empty($importData->getData()) || empty($importData->getHeaders())) {
            throw new \Exception('No data found in file');
        }

        return $importData;
    }

    private function compareKeysWithPreviousImport(array $headers): array {
        $keysPath = $this->lastImportKeysPath();

        $lastImport = file_get_contents($keysPath);
        $lastImportKeys = json_decode($lastImport, true);

        $newKeys = array_diff($headers, $lastImportKeys);
        $removedKeys = array_diff($lastImportKeys, $headers);

        return [
            $newKeys, $removedKeys,
        ];
    }

    private function saveNewKeys(array $keys) {
        $keysPath = $this->lastImportKeysPath();
        file_put_contents($keysPath, json_encode($keys));
    }

    private function lastImportKeysPath() {
        $ebayDir = System::getContainer()->getParameter('kernel.project_dir').'/var/ebay';

        if (!is_dir($ebayDir)) {
            mkdir($ebayDir);
        }

        if (!file_exists($ebayDir.'/last_import_keys.json')) {
            file_put_contents($ebayDir.'/last_import_keys.json', json_encode([]));
        }

        return realpath($ebayDir.'/last_import_keys.json');
    }
}
