<?php

declare(strict_types=1);

namespace MZoldak\EbayOrdersImport\Class\Importers;

use App\Class\Isotope\Address\AddressCreator;
use App\Class\Isotope\Order\NewOrder;
use App\Class\Isotope\Order\OrderManipulator;
use App\Class\Isotope\OrderItems\OrderItemHandler;
use App\Class\Isotope\OrderItems\OrderItemsManipulator;
use App\Class\Isotope\TaxClass;
use App\Constants\OrderStatus;
use App\Constants\PaymentMethod;
use App\EventListener\Isotope\Order\StatusUpdateEventListener;
use Contao\Database;
use Contao\MemberModel;
use Contao\System;
use Isotope\Isotope;
use Isotope\Model\Address;
use Isotope\Model\Product;
use Isotope\Model\ProductCollection\Order;
use MZoldak\EbayOrdersImport\Contants\Ebay;
use MZoldak\EbayOrdersImport\Service\CSVFieldName;

class OrdersImporter {
    public const DUMMY_PRODUCT = 60;
    private MemberModel $member;
    private array $ordersProducts;
    private array $csvOrders;

    public function __construct(MemberModel $member, array $ordersProducts, array $csvOrders) {
        $this->member = $member;
        $this->ordersProducts = $ordersProducts;
        $this->csvOrders = $csvOrders;
        System::loadLanguageFile('default');
    }

    public function importData() {
        foreach ($this->csvOrders as $salesRecordNumber => $csvOrder) {
            if (Order::findOneBy(Ebay::SALES_RECORD_NUMBER, $salesRecordNumber)) {
                continue;
            }

            $orderData = $csvOrder[0];
            $orderProducts = $this->ordersProducts[$salesRecordNumber];
            $orderProducts = $this->createOrderProducts($orderProducts);

            $order = NewOrder::createCart();

            $billingAddress = $this->createBillingAddress($orderData);
            $shippingAddress = $this->createShippingAddress($orderData);

            $order->setBillingAddress($billingAddress);
            $order->setShippingAddress($shippingAddress);

            $order->member = $this->member->id;
            $order->shipping_id = 5;
            $order->payment_id = PaymentMethod::EBAY;
            $order->config_id = 2;
            $order->currency = 'EUR';
            $order->total = $orderData[CSVFieldName::orderTotal()] ?? 0;
            Isotope::setCart($order);
            (new OrderItemsManipulator($order))->synchronizeOrderItems($orderProducts);

            $order = $order->getDraftOrder();
            foreach ($order->getItems() as $item) {
                $order->deleteItem($item);
            }
            $order->save();
            invade($order)->clearCache();

            $billingAddress->pid = $order->id;
            $billingAddress->save();

            $shippingAddress->pid = $order->id;
            $shippingAddress->save();

            $order->{Ebay::SALES_RECORD_NUMBER} = $salesRecordNumber;
            $order->{Ebay::ORDER_NUMBER} = $orderData[CSVFieldName::orderNumber()];
            $order->delivery_price_tax = 'own';
            $order->own_delivery_price_tax = $orderData[CSVFieldName::orderDeliveryPrice()] ?? 0;
            $order->language = 'de';
            $order->date_paid = $orderData[CSVFieldName::orderDatePaid()] ?? null;
            $order->order_status = OrderStatus::WAITING_FOR_SEND;
            $order->is_ebay_order = true;

            (new OrderItemsManipulator($order))->synchronizeOrderItems($orderProducts);
            $order->save();
            invade($order)->clearCache();
            (new OrderManipulator())->synchroniseAll($order);
            StatusUpdateEventListener::generateInvoiceId($order);
            $order->save();

            Database::getInstance()->prepare('UPDATE tl_iso_product_collection SET locked = ? WHERE id = ?')->execute($orderData[CSVFieldName::orderLocked()], $order->id);
            $createdOrdersIds[] = $order->id;
        }

        return $createdOrdersIds;
    }

    private function createShippingAddress(array $orderData): Address {
        $addressCreator = new AddressCreator();
        $fullName = explode(" ", $orderData[CSVFieldName::shippingAddressFullname()]);

        return $addressCreator->create([
            'firstname' => array_shift($fullName) ?? '',
            'lastname' => implode(' ', $fullName) ?? '',
            'country' => $orderData[CSVFieldName::shippingAddressCountry()],
            'city' => $orderData[CSVFieldName::shippingAddressCity()],
            'postal' => $orderData[CSVFieldName::shippingAddressPostal()],
            'street_1' => $orderData[CSVFieldName::shippingAddressStreet1()] ?? '',
            'street_2' => $orderData[CSVFieldName::shippingAddressStreet2()] ?? '',
            'street_3' => $orderData[CSVFieldName::shippingAddressStreet3()] ?? '',
            'phone' => $orderData[CSVFieldName::shippingAddressPhone()] ?? '',
            'email' => $orderData[CSVFieldName::shippingAddressEmail()] ?? '',
        ]);
    }

    private function createBillingAddress(array $orderData): Address {
        $addressCreator = new AddressCreator();
        $fullName = explode(' ', $orderData[CSVFieldName::billingAddressFullname()]);

        return $addressCreator->create([
            'firstname' => array_shift($fullName) ?? '',
            'lastname' => implode(' ', $fullName) ?? '',
            'country' => $orderData[CSVFieldName::billingAddressCountry()],
            'city' => $orderData[CSVFieldName::billingAddressCity()],
            'postal' => $orderData[CSVFieldName::billingAddressPostal()],
            'street_1' => $orderData[CSVFieldName::billingAddressStreet1()] ?? '',
            'street_2' => $orderData[CSVFieldName::billingAddressStreet2()] ?? '',
            'street_3' => $orderData[CSVFieldName::billingAddressStreet3()] ?? '',
            'phone' => $orderData[CSVFieldName::billingAddressPhone()] ?? '',
            'email' => $orderData[CSVFieldName::billingAddressEmail()] ?? '',
        ]);
    }

    /**
     * @param array<string, mixed> $orderItems
     *
     * @return OrderItemHandler[]
     */
    private function createOrderProducts(array $orderItems) {
        return array_map(function ($product) {
            $productId = empty($product['product_id']) ? self::DUMMY_PRODUCT : $product['product_id'];
            $shopProduct = Product::findByPk($productId);
            $taxClass = TaxClass::getByTaxRate((int) $product['tax']);

            $newOrderItem = new OrderItemHandler($shopProduct, $taxClass);
            $newOrderItem->setPrice($product['unit_price']);
            $newOrderItem->setQuantity((int) $product['quantity']);

            if (self::DUMMY_PRODUCT == $productId) {
                $newOrderItem->setName((string) $product['product']);
            }

            return $newOrderItem;
        }, $orderItems);
    }
}
