<?php

declare(strict_types=1);

namespace MZoldak\EbayOrdersImport\Class;

use Contao\MemberModel;
use Isotope\Model\Product;
use Isotope\Model\ProductCollection\Order;
use MZoldak\EbayOrdersImport\Class\Importers\OrdersImporter;
use MZoldak\EbayOrdersImport\Contants\Ebay;
use MZoldak\EbayOrdersImport\Service\CSVFieldName;

class FEDataSorter {
    public static function sort(array $csvData) {
        [$existingMembers, $membersToCreate] = self::usersPreview($csvData);
        [$existingOrders, $ordersToCreate] = self::ordersPreview($csvData);

        $usersAssignedOrders = self::usersAssignedOrders($csvData, $ordersToCreate);

        return [
            'existingMembers' => $existingMembers,
            'membersToCreate' => $membersToCreate,
            'existingOrders' => $existingOrders,
            'ordersToCreate' => array_unique($ordersToCreate),
            'usersAssignedOrders' => $usersAssignedOrders,
        ];
    }

    private static function usersPreview(array $csvData) {
        $ebayUsernames = array_map(fn ($row) => $row[CSVFieldName::username()], $csvData);

        $existingMembers = array_map(function ($username) {
            return MemberModel::findOneBy('ebay_username', $username);
        }, $ebayUsernames);

        $existingMembers = array_filter($existingMembers, fn ($member) => $member);

        if (empty($existingMembers)) {
            return [[], $ebayUsernames];
        }

        $existingMembersUsernames = array_map(fn ($member) => $member->ebay_username, $existingMembers);

        $membersToCreate = array_filter($ebayUsernames, fn ($username) => !in_array($username, $existingMembersUsernames));

        return [$existingMembersUsernames, $membersToCreate];
    }

    private static function ordersPreview(array $csvData) {
        $ebayOrderNumbers = array_map(fn ($row) => $row[CSVFieldName::orderSalesRecordNumber()], $csvData);

        $existingOrders = array_map(fn ($orderNumber) => Order::findOneBy(Ebay::SALES_RECORD_NUMBER, $orderNumber), $ebayOrderNumbers);

        $existingOrders = array_filter($existingOrders, fn ($order) => $order);

        if (!$existingOrders) {
            return [[], $ebayOrderNumbers];
        }

        $existingOrdersNumbers = array_map(fn ($order) => $order->ebay_sales_record_number, $existingOrders);

        $ordersToCreate = array_filter($ebayOrderNumbers, fn ($orderNumber) => !in_array($orderNumber, $existingOrdersNumbers));

        return [$existingOrdersNumbers, $ordersToCreate];
    }

    private static function usersAssignedOrders(array $csvData, array $newOrdersIds) {
        $usersAssignedOrders = [];

        $csvData = array_filter($csvData, fn ($row) => !empty($row[CSVFieldName::offerTitle()]));

        foreach ($csvData as $row) {
            
            if(!in_array($row[CSVFieldName::orderSalesRecordNumber()], $newOrdersIds)) {
                continue;
            }

            $product = Product::findBySku($row[CSVFieldName::productSku()]);

            $titleQuantity = self::titleQuantity((string) $row[CSVFieldName::offerTitle()]);
            $itemsQuantity = $titleQuantity * $row[CSVFieldName::productQuantity()];
            
            $usersAssignedOrders[$row[CSVFieldName::username()]][$row[CSVFieldName::orderSalesRecordNumber()]][] = [
                'product_id' => $product?->id,
                Ebay::SALES_RECORD_NUMBER => $row[CSVFieldName::orderSalesRecordNumber()],
                Ebay::ORDER_NUMBER => $row[CSVFieldName::orderNumber()],
                'sku' => $row[CSVFieldName::productSku()],
                'product_exists' => !empty($product),
                'product' => $product?->name ?? $row[CSVFieldName::offerTitle()],
                'tax' => $row[CSVFieldName::productTaxRate()],
                'quantity' => (int) ($product ? $itemsQuantity : $row[CSVFieldName::productQuantity()]),
                'total' => $row[CSVFieldName::orderTotal()],
                'unit_price' => $product ? $row[CSVFieldName::piecePrice()] / $titleQuantity : $row[CSVFieldName::piecePrice()],
                'ordered_at' => $row[CSVFieldName::orderLocked()],
            ];
        }

        return $usersAssignedOrders;
    }

    private static function titleQuantity(string $dealTitle) {
        if (preg_match('/\d+\s?x/', $dealTitle)) {
            preg_match('/\d+\s?x/', $dealTitle, $matches);

            // dump($matches);
            return (int) $matches[0];
        }

        return 1;
    }
}
