<?php

declare(strict_types=1);

namespace MZoldak\EbayOrdersImport\Class;

class CSVDataProcessor {
    private array $data;

    public function __construct(
        private string $filePath,
    ) {
        $this->data = $this->readData();
    }

    public function getData() {
        return $this->data;
    }

    public function getHeaders() {
        return array_keys($this->data[0]);
    }

    private function readData(): array {
        $fileContent = file($this->filePath);
        if (false === $fileContent) {
            throw new \RuntimeException('Failed to read the file: '.$this->filePath);
        }
        $data = array_map(fn ($d) => str_getcsv($d, ';', '"'), $fileContent);
        $data = $this->removeUnnaturalRows($data, [0, 2]);

        $headers = array_shift($data);

        $data = $this->updateTypes($data);

        return $this->mapHeaders($data, $headers);
    }

    private function mapHeaders(array $data, array $headers): array {
        return array_map(function ($row) use ($headers) {
            return array_combine($headers, $row);
        }, $data);
    }

    /**
     * Removes rows that most probably do not contain valid data.
     *
     * At index 1 all rows should be empty
     * Last 3 Indexes was almost empty, and had unneeded for import data
     * At the moment of implementation, rows had 71 indexes
     */
    private function removeUnnaturalRows(array $data, array $rowIndexes): array {
        foreach ($rowIndexes as $rowIndex) {
            $row = array_filter($data[$rowIndex], fn ($value) => '' != $value);

            if (count($row) > 30) {
                throw new \Error('Row '.$rowIndex.' should be empty or contain unsufficient data for import, it seems to be long enough to load CSV File structure has changed and code has to be adjusted!');
            }

            unset($data[$rowIndex]);
        }

        return array_filter($data, function ($row) {
            return count($row) > 20;
        });
    }

    private function updateTypes(array $data): array {
        return array_map(function ($row) {
            return array_map(function ($value) {
                return $this->processValue($value);
            }, $row);
        }, $data);
    }

    private function processValue(mixed $value) {
        if (preg_match('/\d+[,]\d{1,2}\s?€/', $value, $matches)) {
            $price = str_replace('€', '', $matches[0]);
            $price = str_replace(',', '.', $price);

            return (float) $price;
        }

        if (preg_match('/\d+[,]\d{1,2}\s?%?/', $value)) {
            preg_match('/\d+[,]\d{1,2}\s?%?/', $value, $matches);
            $precentValue = str_replace('%', '', $matches[0]);
            $precentValue = str_replace(',', '.', $precentValue);

            return (float) $precentValue;
        }

        if (preg_match('/\d{2}-\S{3}-\d{2}/u', $value)) {
            $value = str_replace(
                ['Mär', 'Mai', 'Okt', 'Dez'],
                ['Mar', 'May', 'Oct', 'Dec'],
                $value
            );
        }

        if (0 === date_parse_from_format('d-M-y', $value)['error_count']) {
            return date_create_from_format('d-M-y', $value)->getTimestamp();
        }

        return $value;
    }
}
